<?php
/**
 * Markdown 解析器类
 * 用于将 Markdown 转换为 HTML
 *
 * @package WP_Markdown_Cleaner
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * WP_MD_Cleaner_Parser 类
 */
class WP_MD_Cleaner_Parser {

    /**
     * 将 Markdown 转换为 HTML
     *
     * @param string $text Markdown 文本
     * @return string HTML 内容
     */
    public function parse( $text ) {
        if ( empty( $text ) ) {
            return '';
        }

        // 统一换行符
        $text = str_replace( "\r\n", "\n", $text );
        $text = str_replace( "\r", "\n", $text );

        // 确保以换行符结束
        $text .= "\n";

        // 1. 处理代码块 (避免内部被解析)
        $text = $this->parse_code_blocks( $text );

        // 2. 处理标题
        $text = $this->parse_headings( $text );

        // 3. 处理分隔线
        $text = $this->parse_hr( $text );

        // 4. 处理引用
        $text = $this->parse_blockquotes( $text );

        // 5. 处理列表
        $text = $this->parse_lists( $text );

        // 6. 处理段落 (这是最后一步块级处理)
        $text = $this->parse_paragraphs( $text );

        // 7. 处理行内元素 (在块级元素处理完后进行)
        $text = $this->parse_inline_elements( $text );

        // 8. 还原代码块
        $text = $this->restore_code_blocks( $text );

        return trim( $text );
    }

    /**
     * 代码块占位符映射
     */
    private $code_blocks = array();

    /**
     * 解析代码块并用占位符替换
     */
    private function parse_code_blocks( $text ) {
        // 匹配 ```code```
        $text = preg_replace_callback( '/^```([a-zA-Z0-9]*)\n(.*?)\n```$/ms', array( $this, 'code_block_callback' ), $text );
        return $text;
    }

    private function code_block_callback( $matches ) {
        $id = uniqid( 'md_code_block_' );
        $language = $matches[1] ? ' class="language-' . esc_attr( $matches[1] ) . '"' : '';
        $code = htmlspecialchars( $matches[2] );
        $this->code_blocks[$id] = "<pre><code{$language}>{$code}</code></pre>";
        return "\n\n" . $id . "\n\n";
    }

    private function restore_code_blocks( $text ) {
        foreach ( $this->code_blocks as $id => $code ) {
            $text = str_replace( $id, $code, $text );
            // 处理可能被包裹在 <p> 中的占位符
            $text = str_replace( "<p>$code</p>", $code, $text );
        }
        return $text;
    }

    /**
     * 解析标题
     */
    private function parse_headings( $text ) {
        // ATX headers
        return preg_replace_callback( '/^(#{1,6})\s+(.+?)$/m', function( $matches ) {
            $level = strlen( $matches[1] );
            return "<h{$level}>" . trim( $matches[2] ) . "</h{$level}>";
        }, $text );
    }

    /**
     * 解析分隔线
     */
    private function parse_hr( $text ) {
        return preg_replace( '/^([-*_])\1{2,}$/m', '<hr />', $text );
    }

    /**
     * 解析引用
     */
    private function parse_blockquotes( $text ) {
        // 简单的引用支持
        return preg_replace_callback( '/^(>+)\s+(.+?)$/m', function( $matches ) {
            return "<blockquote>" . trim( $matches[2] ) . "</blockquote>";
        }, $text );
    }

    /**
     * 解析列表
     */
    private function parse_lists( $text ) {
        // 无序列表
        $text = preg_replace_callback( '/((?:^[-*+]\s+.+\n?)+)/m', function( $matches ) {
            $list = $matches[1];
            $list = preg_replace( '/^[-*+]\s+(.+)$/m', '<li>$1</li>', $list );
            return "<ul>\n" . $list . "</ul>\n";
        }, $text );

        // 有序列表
        $text = preg_replace_callback( '/((?:^\d+\.\s+.+\n?)+)/m', function( $matches ) {
            $list = $matches[1];
            $list = preg_replace( '/^\d+\.\s+(.+)$/m', '<li>$1</li>', $list );
            return "<ol>\n" . $list . "</ol>\n";
        }, $text );

        return $text;
    }

    /**
     * 解析段落
     * 任何不以HTML标签开头的行，都被视为段落
     */
    private function parse_paragraphs( $text ) {
        $lines = explode( "\n", $text );
        $out = array();
        
        foreach ( $lines as $line ) {
            $line = trim( $line );
            if ( empty( $line ) ) {
                $out[] = '';
                continue;
            }
            
            // 如果已经是 HTML 标签开头（比如之前解析的 h1, ul 等），则不包裹 p
            if ( preg_match( '/^<(\w+)([^>]*)>/', $line ) ) {
                $out[] = $line;
            } else if ( isset( $this->code_blocks[$line] ) ) {
                $out[] = $line; // 占位符
            } else {
                $out[] = "<p>{$line}</p>";
            }
        }
        
        return implode( "\n", $out );
    }

    /**
     * 解析行内元素
     */
    private function parse_inline_elements( $text ) {
        // 图片 ![alt](url)
        $text = preg_replace( '/!\[([^\]]*)\]\(([^)\s]+)(?:\s+"[^"]*")?\)/', '<img src="$2" alt="$1" />', $text );
        
        // 链接 [text](url)
        $text = preg_replace( '/\[([^\]]+)\]\(([^)\s]+)(?:\s+"[^"]*")?\)/', '<a href="$2">$1</a>', $text );
        
        // 粗体 **text** or __text__
        $text = preg_replace( '/(\*\*|__)(.*?)\1/', '<strong>$2</strong>', $text );
        
        // 斜体 *text* or _text_
        $text = preg_replace( '/(\*|_)(.*?)\1/', '<em>$2</em>', $text );
        
        // 删除线 ~~text~~
        $text = preg_replace( '/~~(.*?)~~/', '<del>$1</del>', $text );
        
        // 行内代码 `text`
        $text = preg_replace( '/`([^`]+)`/', '<code>$1</code>', $text );
        
        return $text;
    }
}
