<?php
/**
 * 核心类
 *
 * @package WP_Markdown_Cleaner
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * WP_MD_Cleaner_Core 类
 */
class WP_MD_Cleaner_Core {

    /**
     * 单例实例
     *
     * @var WP_MD_Cleaner_Core
     */
    private static $instance = null;

    /**
     * 获取单例实例
     *
     * @return WP_MD_Cleaner_Core
     */
    public static function instance() {
        if ( null === self::$instance ) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * 构造函数
     */
    private function __construct() {
        $this->init_hooks();
    }

    /**
     * 初始化钩子
     */
    private function init_hooks() {
        // 后台初始化
        if ( is_admin() ) {
            add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_admin_scripts' ) );
            // 在"添加媒体"按钮旁添加按钮
            add_action( 'media_buttons', array( $this, 'add_media_button' ), 15 );
            // 添加页脚 HTML（用于弹窗）
            add_action( 'admin_footer', array( $this, 'add_modal_html' ) );
        }

        // 注册 AJAX 处理：导入 Markdown
        add_action( 'wp_ajax_wp_md_cleaner_import', array( $this, 'ajax_import_markdown' ) );
    }

    /**
     * 在"添加媒体"按钮旁添加按钮
     *
     * @param string $editor_id 编辑器ID
     */
    public function add_media_button( $editor_id ) {
        // 只在主内容编辑器显示
        if ( 'content' !== $editor_id ) {
            return;
        }
        ?>
        <button type="button" id="wp-md-import-btn" class="button wp-md-cleaner-media-btn">
            <span class="dashicons dashicons-upload" style="vertical-align: text-top;"></span>
            <?php esc_html_e( '导入Markdown', 'wp-markdown-cleaner' ); ?>
        </button>
        <?php
    }

    /**
     * 添加弹窗 HTML 到页脚
     */
    public function add_modal_html() {
        global $pagenow;
        if ( ! in_array( $pagenow, array( 'post.php', 'post-new.php' ), true ) ) {
            return;
        }
        ?>
        <div id="wp-md-import-modal" style="display:none;">
            <div class="wp-md-modal-overlay"></div>
            <div class="wp-md-modal-content">
                <div class="wp-md-modal-header">
                    <h3><?php esc_html_e( '导入 Markdown', 'wp-markdown-cleaner' ); ?></h3>
                    <button type="button" class="wp-md-modal-close">&times;</button>
                </div>
                <div class="wp-md-modal-body">
                    <p class="description"><?php esc_html_e( '请在下方粘贴您的 Markdown 内容，点击转换按钮将其插入到编辑器中。', 'wp-markdown-cleaner' ); ?></p>
                    <textarea id="wp-md-input-area" placeholder="# 标题
**粗体**
- 列表项"></textarea>
                </div>
                <div class="wp-md-modal-footer">
                    <button type="button" class="button button-primary button-large" id="wp-md-process-import">
                        <?php esc_html_e( '转换并插入', 'wp-markdown-cleaner' ); ?>
                    </button>
                </div>
            </div>
        </div>
        <?php
    }

    /**
     * 加载后台脚本和样式
     *
     * @param string $hook 当前页面钩子
     */
    public function enqueue_admin_scripts( $hook ) {
        // 只在文章编辑页面加载
        if ( ! in_array( $hook, array( 'post.php', 'post-new.php' ), true ) ) {
            return;
        }

        // 加载样式
        wp_enqueue_style(
            'wp-md-cleaner-admin',
            WP_MD_CLEANER_PLUGIN_URL . 'admin/assets/css/admin.css',
            array(),
            WP_MD_CLEANER_VERSION
        );

        // 加载脚本
        wp_enqueue_script(
            'wp-md-cleaner-import',
            WP_MD_CLEANER_PLUGIN_URL . 'admin/assets/js/import.js',
            array( 'jquery' ),
            WP_MD_CLEANER_VERSION,
            true
        );

        wp_localize_script(
            'wp-md-cleaner-import',
            'wpMdImportData',
            array(
                'ajax_url' => admin_url( 'admin-ajax.php' ),
                'nonce'    => wp_create_nonce( 'wp_md_cleaner_nonce' ),
                'i18n'     => array(
                    'no_content' => __( '请输入 Markdown 内容', 'wp-markdown-cleaner' ),
                    'processing' => __( '正在转换...', 'wp-markdown-cleaner' ),
                    'success'    => __( '导入成功！', 'wp-markdown-cleaner' ),
                    'error'      => __( '转换失败，请重试。', 'wp-markdown-cleaner' ),
                ),
            )
        );
    }

    /**
     * AJAX 处理 Markdown 导入
     */
    public function ajax_import_markdown() {
        // 验证 nonce
        if ( ! check_ajax_referer( 'wp_md_cleaner_nonce', 'nonce', false ) ) {
            wp_send_json_error( array( 'message' => __( '安全验证失败', 'wp-markdown-cleaner' ) ) );
            return;
        }

        // 验证用户权限
        if ( ! current_user_can( 'edit_posts' ) ) {
            wp_send_json_error( array( 'message' => __( '权限不足', 'wp-markdown-cleaner' ) ) );
            return;
        }

        // 获取内容
        $content = isset( $_POST['content'] ) ? wp_unslash( $_POST['content'] ) : '';

        if ( empty( $content ) ) {
            wp_send_json_error( array( 'message' => __( '没有内容', 'wp-markdown-cleaner' ) ) );
            return;
        }

        // 引入解析器并转换
        require_once WP_MD_CLEANER_PLUGIN_DIR . 'includes/class-parser.php';
        
        if ( class_exists( 'WP_MD_Cleaner_Parser' ) ) {
            $parser = new WP_MD_Cleaner_Parser();
            $html = $parser->parse( $content );

            wp_send_json_success( array(
                'html' => $html,
                'message' => __( '转换成功', 'wp-markdown-cleaner' ),
            ) );
        } else {
            wp_send_json_error( array( 'message' => __( '解析器类未找到', 'wp-markdown-cleaner' ) ) );
        }
    }
}
